/* 
 * Copyright (C) 2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: play_item.c,v 1.4 2003/03/02 12:49:32 guenter Exp $
 *
 * playlist item / media mark:
 * an mrl + options (e.g. volume, start time, brightness/contrast ...)
 */

#include <config.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <glib.h>

#include "play_item.h"
#include "utils.h"
#include "playlist.h"

/*
#define LOG
*/

play_item_t *play_item_new (const char *title, const char *mrl, 
			    int start_time) {

  play_item_t *play_item;

  play_item = (play_item_t *) malloc (sizeof (play_item_t));

  play_item->title      = strdup (title);
  play_item->mrl        = strdup (mrl);
  play_item->start_time = start_time,
  play_item->options    = NULL;
  play_item->played     = 0;
  
  return play_item;
}

play_item_t *play_item_copy (play_item_t *src) {

  play_item_t *item;
  GList       *option;

  item = play_item_new (src->title, src->mrl, src->start_time);

  option = src->options;

  while (option) {

    item->options = g_list_append (item->options, strdup (option->data));

    option = g_list_next (option);
  }
  
  return item;
}

void play_item_dispose (play_item_t *play_item) {
  GList *option;

  free (play_item->mrl);
  free (play_item->title);

  option = play_item->options;

  while (option) {

    free (option->data);

    option = g_list_next (option);
  }

  free (play_item);
}

void play_item_add_option (play_item_t *item, const char *option) {

  item->options = g_list_append (item->options, strdup (option));

}

static void printf_depth (FILE *fh, int depth) {
  int i;
  for (i=0; i<depth; i++)
    fprintf (fh, "  ");
}

void play_item_save (play_item_t *item, FILE *fh, int depth) {

  printf_depth (fh, depth); fprintf (fh, "<ENTRY>\n");
  printf_depth (fh, depth); fprintf (fh, "  <TITLE>%s</TITLE>\n", item->title);
  printf_depth (fh, depth); fprintf (fh, "  <REF HREF=\"%s\"/>\n", item->mrl);
  printf_depth (fh, depth); fprintf (fh, "  <TIME start=\"%d\"/>\n", item->start_time);
  printf_depth (fh, depth); fprintf (fh, "</ENTRY>\n");
}

play_item_t *play_item_load (xml_node_t *node) {

  play_item_t *play_item;

  play_item = malloc (sizeof (play_item_t));

  play_item->title      = NULL;
  play_item->options    = NULL;
  play_item->mrl        = NULL;
  play_item->start_time = 0;

  while (node) {

    if (!strcasecmp (node->name, "title")) {
      play_item->title = strdup (node->data);
#ifdef LOG
      printf ("play_item: title = %s\n", play_item->title);
#endif
    } else if (!strcasecmp (node->name, "ref")) {
      play_item->mrl = strdup(xml_parser_get_property (node, "href"));
#ifdef LOG
      printf ("play_item: mrl   = %s\n", play_item->mrl);
#endif
    } else if (!strcasecmp (node->name, "time")) {
      play_item->start_time = xml_parser_get_property_int (node, "start", 0);
#ifdef LOG
      printf ("play_item: start = %d\n", play_item->start_time);
#endif
    } else 
      printf ("play_item: error while loading, unknown node %s\n", node->name);

    node = node->next;
  }

  return play_item;
}

/* gui part */

static GtkWidget *dlg;
static GtkWidget *entry, *mrl_entry, *time_entry;
static int        is_visible;

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (dlg);
  is_visible = 0;

  gtk_main_quit();

  return TRUE;
}

void play_item_init (void) {

  GtkWidget *label, *b;
  GtkWidget *table;

  dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (dlg), "Edit play item...");
  gtk_window_set_default_size (GTK_WINDOW (dlg), 500, 150);
  gtk_signal_connect( GTK_OBJECT (dlg), "delete_event",
		      GTK_SIGNAL_FUNC (close_cb), NULL );

  b = gtk_dialog_add_button (GTK_DIALOG (dlg), GTK_STOCK_CLOSE, 1);
  gtk_signal_connect (GTK_OBJECT(b), "clicked",
		      GTK_SIGNAL_FUNC (close_cb), NULL );

  table = gtk_table_new (3,2,FALSE);

  label = gtk_label_new ("MRL:"); 
  gtk_label_set_justify (GTK_LABEL(label), GTK_JUSTIFY_LEFT);
  gtk_table_attach (GTK_TABLE(table), label, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 2, 2);
  mrl_entry = gtk_entry_new (); 
  gtk_table_attach (GTK_TABLE(table), mrl_entry, 1, 2, 0, 1, GTK_EXPAND|GTK_FILL, GTK_FILL, 0, 2);

  label = gtk_label_new ("Name:"); 
  gtk_label_set_justify (GTK_LABEL(label), GTK_JUSTIFY_LEFT);
  gtk_table_attach (GTK_TABLE(table), label, 0, 1, 1, 2, GTK_FILL, GTK_FILL, 2, 2);
  entry = gtk_entry_new ();
  gtk_table_attach (GTK_TABLE(table), entry, 1, 2, 1, 2, GTK_EXPAND|GTK_FILL, GTK_FILL, 0, 2);

  label = gtk_label_new ("Time:"); 
  gtk_label_set_justify (GTK_LABEL(label), GTK_JUSTIFY_LEFT);
  gtk_table_attach (GTK_TABLE(table), label, 0, 1, 2, 3, GTK_FILL, GTK_FILL, 2, 2);

  time_entry = gtk_entry_new ();
  gtk_table_attach (GTK_TABLE(table), time_entry, 1, 2, 2, 3, GTK_EXPAND|GTK_FILL, GTK_FILL, 0, 2);

  gtk_widget_grab_focus (entry);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox),
		      table, TRUE, TRUE, 0);

  is_visible = 0 ;

}

int play_item_edit (play_item_t *item) {

  gboolean   was_modal;
  char       time_str[256];

  gtk_entry_set_text (GTK_ENTRY (mrl_entry), item->mrl);
  gtk_entry_set_text (GTK_ENTRY (entry), item->title);
  int_to_timestring (item->start_time, time_str, 256);
  gtk_entry_set_text (GTK_ENTRY (time_entry), time_str);

  gtk_widget_show_all (dlg);
  gtk_widget_map (dlg);

  /* run as a modal dialog */

  was_modal = GTK_WINDOW(dlg)->modal;
  if (!was_modal) 
    gtk_window_set_modal(GTK_WINDOW(dlg),TRUE);

  gtk_main();

  item->mrl   = strdup(gtk_entry_get_text (GTK_ENTRY (mrl_entry)));
  item->title = strdup(gtk_entry_get_text (GTK_ENTRY (entry)));
  item->start_time = parse_timestring (gtk_entry_get_text (GTK_ENTRY (time_entry)));

  return 0;
}

