/*
 * Copyright (C) 2002-2003 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: mediamarks.c,v 1.18 2003/03/08 15:50:36 guenter Exp $
 *
 * mediamarks - load
 *            - save
 *            - edit
 * functions
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>

#include "globals.h"
#include "utils.h"
#include "gtkxine.h"
#include "actions.h"
#include "mediamarks.h"
#include "playlist.h"
#include "play_item.h"

#define PLACEHOLDER "(---)"

static GtkTreeStore *mm_store;
static GtkWidget    *manage_dlg, *cat_dlg, *cat_entry;
static GtkTreeIter   cat_iter;
static int           is_visible;

static void menu_cb (GtkWidget* widget, gpointer data) {

  play_item_t *item_orig = (play_item_t *) data;

  if (item_orig) {
    play_item_t *item;
    int          pos;

    item = play_item_copy (item_orig);

    pos = playlist_add (item);
    playlist_play (pos);
  }
}

static void gen_items (GtkMenuShell *submenu, GtkTreeIter *iter) {

  int pos;

  pos = 4;

  do {

    GValue       v;
    play_item_t *play_item;
    GtkWidget    *item;
    char         *id;
      
    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      iter, 0, &v);
    id = g_value_peek_pointer (&v);

    if (id && strcmp (id, PLACEHOLDER)) {
      item = gtk_menu_item_new_with_label (id);
      g_value_unset (&v);

      memset (&v, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
				iter, 2, &v);
      play_item = g_value_peek_pointer (&v);
      
      if (play_item) {
	
	gtk_signal_connect (GTK_OBJECT (item), "activate", 
			    GTK_SIGNAL_FUNC (menu_cb), (gpointer) play_item);
	
      } else {
	
	GtkWidget   *submenu;
	GtkTreeIter  children;
	
	submenu = gtk_menu_new ();
	
	if (gtk_tree_model_iter_children (GTK_TREE_MODEL(mm_store), &children, iter))
	  gen_items (GTK_MENU_SHELL(submenu), &children);
	
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), submenu);
	
      }
      
      gtk_menu_shell_insert (submenu, item, pos);
      gtk_widget_show(item);

      pos++;
    }

    g_value_unset (&v);
      
  } while (gtk_tree_model_iter_next (GTK_TREE_MODEL(mm_store), iter));
}

static void update_menu (GtkMenuShell *submenu) {

  GList        *olditems;
  GtkWidget    *item;
  GtkTreeIter   iter;

  /*
   * remove old items (if any)
   */

  olditems = gtk_container_children (GTK_CONTAINER(submenu));
  while ( (item = (GtkWidget *) g_list_nth_data (olditems, 4)) ) {
    gtk_container_remove (GTK_CONTAINER (submenu), item);
    olditems = gtk_container_children (GTK_CONTAINER (submenu));
  }

  /*
   * add current items
   */

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (mm_store), &iter))
    gen_items (submenu, &iter);


}

/* 
 * support new xml based mediamarks file format
 */

static void get_items (xml_node_t *node, int depth, GtkTreeIter *parent) {

  while (node) {

    if (!strcasecmp (node->name, "sub")) {

      GtkTreeIter iter, placeholder;
      char       *id;

      id = xml_parser_get_property (node, "name");

      if (depth)
	gtk_tree_store_append (mm_store, &iter, parent);
      else
	gtk_tree_store_append (mm_store, &iter, NULL);
      gtk_tree_store_set (mm_store, &iter, 0, id, -1);
      gtk_tree_store_append (mm_store, &placeholder, &iter);
      gtk_tree_store_set (mm_store, &placeholder, 0, PLACEHOLDER, -1);

      get_items (node->child, depth+1, &iter);

    } else if (!strcasecmp (node->name, "entry")) {
      play_item_t *play_item = play_item_load (node->child);
      GtkTreeIter  iter;

      if (depth)
	gtk_tree_store_append (mm_store, &iter, parent);
      else
	gtk_tree_store_append (mm_store, &iter, NULL);
      gtk_tree_store_set (mm_store, &iter, 0, play_item->title, 1, 
			  play_item->mrl, 2, play_item, -1);

    } else {
      printf ("mediamarks: error, unknown node type %s\n", node->name);
    }

    node = node->next;
  }
}

static void load_new_file (xml_node_t *node) {

  if (!strcasecmp (node->name, "gxinemm")) {

    get_items (node->child, 0, NULL);
  } else {
    printf ("mediamarks: ERROR, root node must be GXINEMM\n");
  }
}

/*
 * legacy support for old mediamarks file format
 */

static char * read_line (FILE *f) {
  char  str[1024];
  char *str2;
  int   len;

  str2 = fgets (str, 1024, f);

  if (!str2)
    return NULL;

  len = strlen (str);
  if ( (len>0) && (str[len-1] == 10))
    str[len-1] = 0;

  return strdup (str);
}

static void load_old_file (char *fname) {

  FILE        *f;
  int          depth;
  GtkTreeIter  parent;
  GtkTreeIter  iter;

  gtk_tree_model_get_iter_first ( GTK_TREE_MODEL (mm_store), &parent);

  f = fopen (fname, "r");
  if (f) {
    
    depth = 0;

    while (1) {
      
      char *id, *str;

      id = read_line (f);
      if (!id)
	break;

      if (!strncmp (id, "<sub>", 5)) {

	GtkTreeIter placeholder;
	/* sub-tree starts here */

	if (depth)
	  gtk_tree_store_append (mm_store, &iter, &parent);
	else
	  gtk_tree_store_append (mm_store, &iter, NULL);
	gtk_tree_store_set (mm_store, &iter, 0, &id[5], -1);
	gtk_tree_store_append (mm_store, &placeholder, &iter);
	gtk_tree_store_set (mm_store, &placeholder, 0, PLACEHOLDER, -1);

	memcpy (&parent, &iter, sizeof (iter));
	depth++;

      } else if (!strncmp (id, "</sub>", 6)) {
      
	/* sub-tree ends here */

	gtk_tree_model_iter_parent (GTK_TREE_MODEL (mm_store), 
				    &iter, &parent);
	depth--;
	if (depth<0) {

	  printf ("mediamarks: mediamarks file %s corrupted, ignoring rest of file\n",
		  fname);
	  return;
	}
	memcpy (&parent, &iter, sizeof (iter));

      } else { 
	
	play_item_t *play_item;
	char        *mrl;
	int          time;

	mrl = read_line (f);
	if (!mrl)
	  break;

	str = read_line (f);
	if (!str)
	  break;
	sscanf (str, "%d", &time);

	play_item = play_item_new (id, mrl, time);

	if (depth)
	  gtk_tree_store_append (mm_store, &iter, &parent);
	else
	  gtk_tree_store_append (mm_store, &iter, NULL);
	gtk_tree_store_set (mm_store, &iter, 0, id, 1, mrl, 2, play_item, -1);
      }

      /* discard newline */
      str = read_line (f);
      if (!str)
	break;
    }
    fclose (f);
  } else {

    printf ("mediamarks: load failed!\n");

  }
}

static void win_delete_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (widget);
  is_visible = 0;
}

static gboolean close_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (manage_dlg);
  is_visible = 0;

  return TRUE;
}

static void cat_apply (void) {
  const char *name = gtk_entry_get_text (GTK_ENTRY(cat_entry));

  gtk_tree_store_set (mm_store, &cat_iter, 0, name, -1);
}

static void cat_delete_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (widget);
  cat_apply ();
}

static void cat_close_cb (GtkWidget* widget, gpointer data) {
  gtk_widget_hide (data);
  cat_apply ();
}

static void edit_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = (GtkTreeView *) data;
  GtkTreeSelection  *sel;
  GtkTreeIter        iter;  

  sel = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v;
    play_item_t *play_item;

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
    if (play_item) {
#ifdef LOG
      printf ("mediamarks: got a play item\n");
#endif
      
      play_item_edit (play_item);

#ifdef LOG
      printf ("mediamarks: play item title after edit : %s\n",
	      play_item->title);
#endif

      gtk_tree_store_set (mm_store, &iter, 
			  0, play_item->title, 
			  1, play_item->mrl, 2, play_item, -1);


    } else {
      GValue       v;
      char        *name;

      memcpy (&cat_iter, &iter, sizeof (cat_iter));

      memset (&v, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
				&iter, 0, &v);
      name = (char *) g_value_peek_pointer (&v);
      gtk_entry_set_text (GTK_ENTRY (cat_entry), name);
      g_value_unset (&v);

      gtk_widget_show_all (cat_dlg);
      gtk_widget_map (cat_dlg);
      gtk_widget_grab_focus (cat_entry);
    }
    g_value_unset (&v);
  }
}

static void delete_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = (GtkTreeView *) data;
  GtkTreeSelection  *sel;
  GtkTreeIter        iter;  

  sel = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v;
    char        *id;

    memset (&v, 0, sizeof (GValue));

    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      &iter, 0, &v);
    id = (char *) g_value_peek_pointer (&v);

    if (!strcmp (id, PLACEHOLDER)) {
      g_value_unset (&v);
      return;
    }

    g_value_unset (&v);

    gtk_tree_store_remove (mm_store, &iter);
  }
}

static void new_category_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = (GtkTreeView *) data;
  GtkTreeSelection  *sel;
  GtkTreeIter        iter, parent;  
  int                in_root;

  /*
   * find out where to create the new section
   */

  in_root = 1;
  sel     = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v;
    play_item_t *play_item;

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      &iter, 0, &v);

    g_value_unset (&v);

    /*
     * is this a section or a play item ?
     */

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
    if (play_item) {
      
      if (!gtk_tree_model_iter_parent (GTK_TREE_MODEL (mm_store), 
				       &parent, &iter)) {
	in_root = 1;
      } else
	in_root = 0;
    } else {
      GtkTreePath *path;

      path = gtk_tree_model_get_path (GTK_TREE_MODEL (mm_store), &iter);
      
      if (gtk_tree_view_row_expanded (tree, path)) {

	in_root = 0;
	memcpy (&parent, &iter, sizeof (iter));

      } else {

	if (!gtk_tree_model_iter_parent (GTK_TREE_MODEL (mm_store), 
					 &parent, &iter)) {
	  in_root = 1;
	} else
	  in_root = 0;
      }
      
      gtk_tree_path_free (path);
    }
    g_value_unset (&v);
  } else {
    in_root = 1;
  }

  /*
   * create new category
   */

  if (in_root)
    gtk_tree_store_append (mm_store, &cat_iter, NULL);
  else
    gtk_tree_store_append (mm_store, &cat_iter, &parent);
  gtk_tree_store_set (mm_store, &cat_iter, 0, "New Category", -1);
  /* create dummy entry so d&d works */
  gtk_tree_store_append (mm_store, &iter, &cat_iter);
  gtk_tree_store_set (mm_store, &iter, 0, PLACEHOLDER, -1);
  
  /*
   * show dialog, ask for section name
   */

  gtk_entry_set_text (GTK_ENTRY (cat_entry), "New Category");
  gtk_widget_show_all (cat_dlg);
  gtk_widget_map (cat_dlg);
  gtk_widget_grab_focus (cat_entry);
}

static void tree_changed_cb (GtkTreeModel *treemodel,
			     GtkTreePath *arg1,
			     gpointer user_data) {
  update_menu (media_menu);
}

static int load_mediamarks (gchar *fname) {

  char                 *mmfile;

  mmfile = read_entire_file_ascii (fname);

  if (mmfile) {
    xml_node_t *node;

    xml_parser_init (mmfile, strlen (mmfile), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node)<0) {
      load_old_file (fname);
    } else 
      load_new_file (node);

    free (mmfile);
  } else
    return 0;

  return 1;
}

void mm_import (void) {

  gchar *fname;

  fname = modal_file_dialog ("Select mediamarks file to load...", NULL);

  if (fname) {

    if (!load_mediamarks (fname))
      display_error ("Mediamarks load failed",
		     "Couldn't load mediamarks file"
		     "%s", fname);
  }
}

void mediamarks_init (void) {

  GtkWidget            *tree, *b, *scrolled_window, *hbox;
  GtkWidget            *button, *label;
  gchar                *fname;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;
   
  /*
   * init tree store
   */
 
  mm_store = gtk_tree_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

  /* 
   * now parse the mediamarks file
   */

  fname = g_strconcat(g_get_home_dir(), "/.gxine/mediamarks", NULL);

  if (!load_mediamarks (fname)) {
    /* load sample mediamarks */

    g_free (fname);

    fname = g_strconcat (GXINE_MISCDIR, "/mediamarks", NULL);
    load_mediamarks (fname);
  }

  g_free(fname);

  g_signal_connect (G_OBJECT (mm_store), "row-changed",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "row-deleted",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "row-has-child-toggled",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "row-inserted",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "rows-reordered",
		    G_CALLBACK (tree_changed_cb), NULL);

  update_menu (media_menu);


  /*
   * create (for now invisible) mm_manage dialog
   */

  manage_dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (manage_dlg), "Manage Mediamarks...");
  gtk_window_set_default_size (GTK_WINDOW (manage_dlg), 400, 250);
  g_signal_connect( GTK_OBJECT (manage_dlg), "delete_event",
		      G_CALLBACK (win_delete_cb), NULL );
  b = gtk_dialog_add_button (GTK_DIALOG (manage_dlg), GTK_STOCK_CLOSE, 1);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK (close_cb),
		    manage_dlg);

  /* add a nice tree view widget here */

  tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL(mm_store));  
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree), TRUE);

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("Mediamarks",
						     cell,
						     "text", 0,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree),
			       GTK_TREE_VIEW_COLUMN (column));

  column = gtk_tree_view_column_new_with_attributes ("mrl",
						     cell,
						     "text", 1,
						     NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree),
			       GTK_TREE_VIEW_COLUMN (column));
  
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tree), TRUE);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(manage_dlg)->vbox), scrolled_window,
		      TRUE, TRUE, 2);

  /* add edit button bar */
  hbox = gtk_hbox_new (2, 2);

  button = gtk_button_new_with_label ("New Category...");
  g_signal_connect (button, "clicked", G_CALLBACK(new_category_cb), tree);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 2);  

  button = gtk_button_new_with_label ("Edit...");
  g_signal_connect (button, "clicked", G_CALLBACK(edit_cb), tree);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 2);  

  button = gtk_button_new_with_label ("Delete");
  g_signal_connect (button, "clicked", G_CALLBACK(delete_cb), tree);
  gtk_box_pack_start (GTK_BOX (hbox), button, FALSE, FALSE, 2);  

  gtk_box_pack_end (GTK_BOX(GTK_DIALOG(manage_dlg)->vbox), hbox,
		    FALSE, FALSE, 2);

  is_visible = FALSE;

  /*
   * create (for now invisible) category name entry dialog
   */

  cat_dlg = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (cat_dlg), "Enter category name...");
  g_signal_connect( GTK_OBJECT (cat_dlg), "delete_event",
		      G_CALLBACK (cat_delete_cb), NULL );

  b = gtk_dialog_add_button (GTK_DIALOG (cat_dlg), GTK_STOCK_CLOSE, 1);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		      G_CALLBACK (cat_close_cb),
		      cat_dlg);

  /* all we have is a simple text entry */

  hbox = gtk_hbox_new (2, 2);

  label = gtk_label_new ("Category name:");

  gtk_box_pack_start (GTK_BOX (hbox), label,
		      FALSE, FALSE, 0);
  
  cat_entry = gtk_entry_new (); 
  gtk_signal_connect (GTK_OBJECT(cat_entry), "activate",
		      GTK_SIGNAL_FUNC (cat_close_cb),
		      cat_dlg);  
  gtk_box_pack_start (GTK_BOX (hbox), cat_entry,
		      TRUE, TRUE, 0);

  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (cat_dlg)->vbox), hbox,
		      TRUE, TRUE, 0);
}

void mm_add_show () {

  play_item_t *play_item;
  GtkTreeIter  iter;
  int          pos, pos_time, len;

  play_item = play_item_copy (playlist_get_item (playlist_get_list_pos()));

  gtk_xine_get_pos_length (GTK_XINE (gtx), 
			   &pos, &pos_time, &len);

  play_item->start_time = pos_time;
  play_item_edit (play_item);
  gtk_tree_store_append (mm_store, &iter, NULL);
  gtk_tree_store_set (mm_store, &iter, 0, play_item->title, 
		      1, play_item->mrl, 2, play_item, -1);
}


void mm_manage_show () {

  if (is_visible) {
    is_visible = FALSE;
    gtk_widget_hide (manage_dlg);
  } else {
    is_visible = TRUE;
    gtk_widget_show_all (manage_dlg);
    gtk_widget_map (manage_dlg);
  }
}

static void print_depth (FILE *f, int depth) {
  depth += 2;

  while (depth) {
    fprintf (f, " ");
    depth --;
  }
}

static void save_items (FILE *f, int depth, GtkTreeIter *iter) {

  do {
    
    GValue        v;
    play_item_t  *play_item;
    char         *id;
      
    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      iter, 0, &v);
    id = g_value_peek_pointer (&v);

    if (id && strcmp (id, PLACEHOLDER)) {

      GValue vi;

      memset (&vi, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
				iter, 2, &vi);
      play_item = g_value_peek_pointer (&vi);
      
      if (play_item) {
	
	play_item_save (play_item, f, depth);
	
      } else {

	GtkTreeIter children;

	print_depth (f, depth);
	fprintf (f, "<SUB NAME=\"%s\">\n", id);
	
	if (gtk_tree_model_iter_children (GTK_TREE_MODEL(mm_store), &children, iter))
	  save_items (f, depth+2, &children);
	
	print_depth (f, depth);
	fprintf (f, "</SUB>\n");
	
      }
      
      g_value_unset (&vi);
    }

    g_value_unset (&v);
      
  } while (gtk_tree_model_iter_next (GTK_TREE_MODEL(mm_store), iter));
}

void mm_save (void) {

  FILE    *f;
  gchar   *fname;

  fname = g_strconcat(g_get_home_dir(), "/.gxine/mediamarks", NULL);

  f = fopen (fname, "w") ;

  if (f) {

    GtkTreeIter iter;

    fprintf (f, "<GXINEMM VERSION=\"1\">\n");

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (mm_store), &iter))
      save_items (f, 0, &iter);

    fprintf (f, "</GXINEMM>\n");

    fclose (f);
  } else
    printf ("mediamarks: save to '%s' failed!\n", fname);

  g_free(fname);
}
