/*
 * Copyright (C) 2002-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: settings.c,v 1.18 2004/12/12 20:34:19 dsalt Exp $
 *
 * settings implementation
 */

#include "globals.h"

#include <X11/Xlib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>
#include <stdio.h>
#include <stdlib.h>

#include "settings.h"
#include "utils.h"

struct slider_s {
  const char *label;
  control_adjustment_t type;
};

struct slider_window_s {
  const gchar		*title;
  const struct slider_s *sliders;
  GtkWidget		*window;
  gboolean		 is_visible;
};

static const struct slider_s settings_sliders[] = {
  { N_("Volume"),	Control_VOLUME },
  { N_("Compressor"),	Control_COMPRESSOR },
  { N_("Amplifier"),	Control_AMPLIFIER },
  { N_("A/V sync"),	Control_AV_SYNC },
  { N_("Hue"),		Control_HUE },
  { N_("Saturation"),	Control_SATURATION },
  { N_("Contrast"),	Control_CONTRAST },
  { N_("Brightness"),	Control_BRIGHTNESS },
  { NULL }
};

static struct slider_window_s settings_window = {
  N_("Settings"), settings_sliders, NULL, FALSE
};

#define SLIDERS_SET(FUNC) \
  static gboolean sliders_##FUNC (const struct slider_window_s *window) \
  { \
    const struct slider_s *slider = window->sliders - 1; \
    while ((++slider)->label) \
      ui_##FUNC##_control_adjustment (slider->type); \
    return FALSE; \
  }

#define JSFUNC(OBJ,FUNC) \
  static JSBool js_##OBJ##_##FUNC (JSContext *cx, JSObject *obj, uintN argc, \
				   jsval *argv, jsval *rval) \
  { \
    se_log_fncall (#OBJ"_"#FUNC); \
    sliders##_##FUNC (&OBJ##_window); \
    return JS_TRUE; \
  }

#define JSDECL(OBJ,FUNC) \
  { #OBJ"_"#FUNC, js_##OBJ##_##FUNC, 0, 0,SE_GROUP_DIALOGUE, NULL, NULL }

static void sliders_show (struct slider_window_s *window)
{
  window->is_visible = !window->is_visible;
  if (window->is_visible)
    gtk_widget_show_all (window->window);
  else
    gtk_widget_hide (window->window);
}

SLIDERS_SET(revert)
SLIDERS_SET(reset)
SLIDERS_SET(clear)

static void response_cb (GtkDialog *dbox, gint response, gpointer data)
{
  struct slider_window_s *window = data;
  switch (response)
  {
  case GTK_RESPONSE_CANCEL:
    sliders_revert (window);
    break;
  case 1:
    sliders_reset (window);
    break;
  default:
    window->is_visible = FALSE;
    gtk_widget_hide (window->window);
  }
}

JSFUNC (settings, show)
JSFUNC (settings, revert)
JSFUNC (settings, reset)
JSFUNC (settings, clear)

struct slider_data_s {
  GtkLabel *label;
  gdouble value;
};

static gboolean slider_set_text (struct slider_data_s *data)
{
  char n[12];
  sprintf (n, "%7.0f", data->value);
  gtk_label_set_text (data->label, n);
  free (data);
  return FALSE;
}

static gchar *slider_format_value (GtkScale *scale, gdouble value,
				   gpointer data)
{
  struct slider_data_s *slider = malloc (sizeof (struct slider_data_s));
  slider->label = data;
  slider->value = value;
  g_idle_add ((GSourceFunc) slider_set_text, slider);
  return g_strdup ("");
}

static void sliders_create (struct slider_window_s *window)
{
  GtkWidget *table;
  int i;

  /* window */

  window->window = gtk_dialog_new_with_buttons (gettext (window->title),
			NULL, 0,
			_("_Default"), 1,
			GTK_STOCK_REVERT_TO_SAVED, GTK_RESPONSE_CANCEL,
			GTK_STOCK_CLOSE, GTK_RESPONSE_DELETE_EVENT,
			NULL);
  gtk_window_set_default_size (GTK_WINDOW (window->window), 500, 150);
  hide_on_delete (window->window, &window->is_visible);
  gtk_signal_connect (GTK_OBJECT(window->window), "response",
		      GTK_SIGNAL_FUNC(response_cb), window);

  /* sliders */

  table = gtk_table_new (3, 2, FALSE);

  for (i = 0; window->sliders[i].label; ++i)
  {
    GtkObject *adj;
    GtkWidget *w, *v;

    w = gtk_label_new (gettext (window->sliders[i].label));
    gtk_misc_set_alignment (GTK_MISC (w), 1, 0.5);
    gtk_table_attach (GTK_TABLE(table), w,
		      0, 1, i, i+1, GTK_FILL, GTK_FILL, 2, 2);

    v = gtk_label_new ("");
    gtk_misc_set_alignment (GTK_MISC (v), 1, 0.5);
    gtk_table_attach (GTK_TABLE(table), v,
		      1, 2, i, i+1, GTK_FILL, GTK_FILL, 2, 2);

    adj = ui_register_control_adjustment (window->sliders[i].type);
    w = gtk_hscale_new (GTK_ADJUSTMENT(adj));
    gtk_scale_set_value_pos (GTK_SCALE(w), GTK_POS_LEFT);
    gtk_scale_set_digits (GTK_SCALE(w), 0);
    g_signal_connect (G_OBJECT(w), "format-value",
		      G_CALLBACK(slider_format_value), v);
    gtk_table_attach (GTK_TABLE(table), w,
		      2, 3, i, i+1, GTK_EXPAND | GTK_FILL, GTK_FILL, 0, 0);
  }

  { /* create an 'invisible' label to set the width of the second column */
    GtkWidget *w = gtk_label_new ("-000000");
    gtk_widget_set_size_request (w, -1, 1);
    gtk_table_attach (GTK_TABLE(table), w,
		      1, 2, i, i+1, GTK_FILL, GTK_FILL, 2, 0);
  }

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(window->window)->vbox), table,
		      TRUE, TRUE, 2);

  sliders_revert (window);
}

void settings_init (void)
{
  static const se_f_def_t defs[] = {
    JSDECL (settings, show),
    JSDECL (settings, revert),
    JSDECL (settings, reset),
    JSDECL (settings, clear),
    { NULL }
  };

  sliders_create (&settings_window);

  se_defuns (gse, NULL, defs);
}
