/*
 * Copyright (C) 2003-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: engine.c,v 1.10 2004/12/03 22:01:21 dsalt Exp $
 *
 * init xine engine, set up script engine
 */

#include "globals.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <stdarg.h>

#include "engine.h"
#include "utils.h"

se_t *gse; /* global script engine */
xine_t *xine; /* global xine instance */
pthread_mutex_t engine_lock;

static void print_cb (void *user_data, char *str, ...)
{
  if (verbosity)
  {
    va_list ap;
    printf (_("engine: < "));
    va_start (ap, str);
    vprintf (str, ap);
  }
}

int engine_exec (const char *cmd, se_print_cb_t cb, void *cb_data) {

  JSInt32 num;
  char *str;

  if (verbosity)
    printf (_("engine: > executing '%s'...\n"), cmd);

  if (!cb) {
    cb = print_cb;
    cb_data = NULL;
  }

  se_eval (gse, cmd, NULL, cb, cb_data);

  if ( (str = se_result_str (gse)) ) {
    cb (cb_data, _("result: "));
    cb (cb_data, str);
    return 1;
  }

  if ( se_result_num (gse, &num) ) {
    char str[32];
    snprintf (str, sizeof (str), _("result: %d"), num);
    cb (cb_data, str);
    return num;
  }

  return 0;
}

int v_engine_exec(const char *command, se_print_cb_t cb, void *cb_data, ...)
{
  char *buf;
  va_list va;
  int ret;

  va_start (va, cb_data);
  buf = g_strdup_vprintf (command, va);
  va_end (va);
  ret = engine_exec (buf, cb, cb_data);
  free (buf);
  return ret;
}

void engine_init (void) {

  char *cfgfilename;
  
  /*
   * some xine engine functions may block for a long
   * time - that's why it is often not called directly
   * but by using an extra thread so the gui will stay
   * responsive
   */

  pthread_mutex_init (&engine_lock, NULL);

  /*
   * init the global xine engine
   */

  xine = xine_new ();
  ui_preferences_register (xine);

  cfgfilename = g_strconcat (g_get_home_dir(), "/.gxine/config", NULL);
  xine_config_load (xine, cfgfilename);
  xine_engine_set_param (xine, XINE_ENGINE_PARAM_VERBOSITY, verbosity);

  xine_init (xine);

  /* init the javascript engine */
  
  gse = se_new ();
}
